// Copyright (c) 1997,2002 by Jim Lynch.
// Copyright (c) 2010-2012,2026 by Philipp Schafft.
// This software comes with NO WARRANTY WHATSOEVER.
//
// This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; version 2 dated June, 1991, or, at your
//    option, any LATER version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this software; see the file COPYING.  If not, write to
//    the Free Software Foundation, 51 Franklin Street, Fifth Floor,
//    Boston, MA 02110-1301, USA.

#include <iostream>
#include "sqlite-stuff.h"


static bool WriteInsertOnly(sqlite3 *db, const std::string &theKey, const std::string &theValue) {
    sqlite3_stmt *stmt;

    if (sqlite3_prepare_v2(db, "INSERT INTO animals (key, value) VALUES (?, ?)", -1, &stmt, NULL) != SQLITE_OK)
        return false;

    if (sqlite3_bind_text(stmt, 1, theKey.c_str(), -1, SQLITE_STATIC) != SQLITE_OK) {
        sqlite3_finalize(stmt);
        return false;
    }

    if (sqlite3_bind_text(stmt, 2, theValue.c_str(), -1, SQLITE_STATIC) != SQLITE_OK) {
        sqlite3_finalize(stmt);
        return false;
    }

    if (sqlite3_step(stmt) != SQLITE_DONE) {
        sqlite3_finalize(stmt);
        return false;
    }

    sqlite3_finalize(stmt);

    return true;
}

bool        Write(sqlite3 *db, const std::string &theKey, const std::string &theValue) {
    sqlite3_stmt *stmt;

    if (sqlite3_prepare_v2(db, "DELETE FROM animals WHERE key = ?", -1, &stmt, NULL) != SQLITE_OK)
        return false;

    if (sqlite3_bind_text(stmt, 1, theKey.c_str(), -1, SQLITE_STATIC) != SQLITE_OK) {
        sqlite3_finalize(stmt);
        return false;
    }

    if (sqlite3_step(stmt) != SQLITE_DONE) {
        sqlite3_finalize(stmt);
        return false;
    }

    sqlite3_finalize(stmt);

    return WriteInsertOnly(db, theKey, theValue);
}

bool        Read(sqlite3 *db, const std::string &theKey, std::string &theValue) {
    sqlite3_stmt *stmt;

    if (sqlite3_prepare_v2(db, "SELECT value FROM animals WHERE key = ?", -1, &stmt, NULL) != SQLITE_OK)
        return false;

    if (sqlite3_bind_text(stmt, 1, theKey.c_str(), -1, SQLITE_STATIC) != SQLITE_OK) {
        sqlite3_finalize(stmt);
        return false;
    }

    if (sqlite3_step(stmt) != SQLITE_ROW) {
        sqlite3_finalize(stmt);
        return false;
    }

    theValue = (const char*)sqlite3_column_text(stmt, 0);

    sqlite3_finalize(stmt);

    return true;
}

sqlite3 *   OpenFile(std::string dbfile) {
    sqlite3_stmt *stmt;
    sqlite3 *ret;

    if (!(dbfile.ends_with(".sqlite") || dbfile.ends_with(".sqlite3"))) {
        dbfile += ".sqlite3";
    }

    if (sqlite3_open(dbfile.c_str(), &ret) != SQLITE_OK) {
        std::cerr << "Error: Opening " << dbfile << " failed" << std::endl;
        exit(1);
    }

    if (sqlite3_prepare_v2(ret, "CREATE TABLE IF NOT EXISTS animals (key VARCHAR(255) NOT NULL PRIMARY KEY, value VARCHAR(255) NOT NULL)", -1, &stmt, NULL) != SQLITE_OK) {
        sqlite3_finalize(stmt);
        std::cerr << "Error: Writing " << dbfile << std::endl;
        exit(1);
    }

    if (sqlite3_step(stmt) != SQLITE_DONE) {
        sqlite3_finalize(stmt);
        std::cerr << "Error: Writing " << dbfile << std::endl;
        exit(1);
    }

    sqlite3_finalize(stmt);

    WriteInsertOnly(ret, "last", "0"); // insert default value.

    return ret;
}

void        CloseFile(sqlite3 *db) {
    sqlite3_close(db);
}

void SetLast(sqlite3 *db, const std::string &newKey) {
    Write(db, "last", newKey);
}

std::string GetLast(sqlite3 *db) {
    std::string result;

    Read(db, "last", result);

    return result;
}

void SetDbType(sqlite3 *db, const std::string &newType, bool plural) {
    Write(db, plural ? "dbtype_plural" : "dbtype_singular", newType);
}

std::string GetDbType(sqlite3 *db, bool plural) {
    std::string result;

    bool rdOK = Read(db, plural ? "dbtype_plural" : "dbtype_singular", result);

    // Test if we got something or use defaults:
    if ( !rdOK || result.length() == 0 ) {
        if (plural) {
            result  = GetDbType(db, false);
            result += "s"; // add plural 's'
        } else {
            return "animal";
        }
    }

    return result;
}

bool IsAnimal(sqlite3 *db, const std::string &theKey) {
    std::string theValue;

    Read(db, theKey, theValue);

    return theValue[0] == 'a';
}
